export class NameSelector {
    constructor(config = {}) {
        this.config = {
            names: ['Alice', 'Bob', 'Charlie', 'David', 'Eve', 'Frank'],
            minPlayers: 1,
            maxPlayers: 4,
            title: 'Select Your Names',
            ...config
        };
        
        this.selectedNames = [];
        this.container = null;
        this.doneButton = null;
        this.counter = null;
    }

    createStyles() {
        return {
            container: `
                background-color: #f8f9fa;
                padding: 25px;
                border-radius: 16px;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
                text-align: center;
                min-width: 360px;
                /* font-family: system-ui, -apple-system, sans-serif; */
            `,
            title: `
                margin-bottom: 15px;
                font-size: 28px;
                color: #2c3e50;
                font-weight: 600;
            `,
            instructions: `
                margin-bottom: 25px;
                color: #6c757d;
                font-size: 16px;
            `,
            grid: `
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 12px;
                margin-bottom: 25px;
            `,
            button: `
                padding: 15px 10px;
                border: 2px solid #e9ecef;
                border-radius: 12px;
                background: white;
                cursor: pointer;
                font-size: 21px;
                font-weight: 500;
                color: #495057;
                -webkit-tap-highlight-color: transparent;
            `,
            counter: `
                margin-bottom: 5px;
                font-weight: 600;
                font-size: 22px;
                color: #495057;
            `,
            doneButton: `
                padding: 12px 35px;
                margin-bottom: 25px;
                background: #6862c7;
                color: white;
                border: none;
                border-radius: 12px;
                cursor: pointer;
                font-size: 18px;
                font-weight: 600;
            `
        };
    }

    updateCounter() {

        this.counter.style.color = this.selectedNames.length > 0 ? '#6862c7' : '#666';        
        this.doneButton.disabled = this.selectedNames.length < this.config.minPlayers;
        
        //console.log(`updateCounter() selectedNames.length = ${this.selectedNames.length}`);

        // Update done button appearance
        if (this.doneButton.disabled) {
            this.doneButton.style.opacity = '0.5';
            this.doneButton.style.cursor = 'not-allowed';
            this.doneButton.style.background = '#cccccc';
        } else {
            this.doneButton.style.opacity = '1';
            this.doneButton.style.cursor = 'pointer';
            this.doneButton.style.background = '#6862c7'; //'#4CAF50';
        }
        this.counter.textContent = `Selected: ${this.selectedNames.length}/${this.config.maxPlayers}`;

        const buttons = this.container.querySelectorAll('.name-button');
        buttons.forEach(btn => {
            if (!this.selectedNames.includes(btn.textContent)) {
                btn.disabled = this.selectedNames.length >= this.config.maxPlayers;
                btn.style.opacity = btn.disabled ? '0.5' : '1';
                btn.style.cursor = btn.disabled ? 'not-allowed' : 'pointer';
            }
        });
    }

    createNameButton(name) {
        const styles = this.createStyles();
        const button = document.createElement('button');
        button.textContent = name;
        button.className = 'name-button';
        button.style.cssText = styles.button;

        button.addEventListener('click', () => {
            if (this.selectedNames.includes(name)) {
                this.selectedNames = this.selectedNames.filter(n => n !== name);
                button.style.borderColor = '#e9ecef';
                button.style.backgroundColor = 'white';
            } else if (this.selectedNames.length < this.config.maxPlayers) {
                this.selectedNames.push(name);
                button.style.borderColor = '#6862c7';
                button.style.backgroundColor = '#d7cbff'; //'#e8f5e9';
            }
            this.updateCounter();
        });

        return button;
    }

    show(targetElement) {
        return new Promise((resolve) => {
            const styles = this.createStyles();
            
            this.container = document.createElement('div');
            this.container.style.cssText = styles.container;

            const title = document.createElement('h2');
            title.textContent = this.config.title;
            title.style.cssText = styles.title;
            this.container.appendChild(title);

/*
            const instructions = document.createElement('p');
            instructions.textContent = `Select between ${this.config.minPlayers} and ${this.config.maxPlayers} names`;
            instructions.style.cssText = styles.instructions;
            this.container.appendChild(instructions);
*/
            this.counter = document.createElement('div');
            this.counter.style.cssText = styles.counter;
            this.container.appendChild(this.counter);

            // done button
            this.doneButton = document.createElement('button');
            this.doneButton.textContent = 'Done';
            this.doneButton.disabled = true;
            this.doneButton.style.cssText = styles.doneButton;
            /*
            if (this.doneButton.disabled) {
                this.doneButton.style.opacity = '0.5';
                this.doneButton.style.cursor = 'not-allowed';
                this.doneButton.style.background = '#cccccc';
            }
*/
            this.doneButton.addEventListener('click', () => {
                if (this.selectedNames.length >= this.config.minPlayers) {
                    const selectedNamesArray = [...this.selectedNames];
                    this.container.remove();
                    resolve(selectedNamesArray);
                }
            });

            this.container.appendChild(this.doneButton);

            const grid = document.createElement('div');
            grid.style.cssText = styles.grid;
            
            this.config.names.forEach(name => {
                grid.appendChild(this.createNameButton(name));
            });
            
            this.container.appendChild(grid);

            targetElement.appendChild(this.container);
            this.updateCounter();
        });
    }
}

// Usage example:
/*
import { NameSelector } from './nameSelector.js';

// Create with default config
const selector = new NameSelector();

// Or create with custom config
const selector = new NameSelector({
    names: ['Alice', 'Bob', 'Charlie', 'David', 'Eve', 'Frank'],
    minPlayers: 2,
    maxPlayers: 3,
    title: 'Select Team Members'
});

// Show the selector in a specific element
const targetElement = document.getElementById('game-container');
const startGame = async () => {
    const selectedNames = await selector.show(targetElement);
    console.log('Selected players:', selectedNames);
};

startGame();
*/