import {
    PLACEHOLDER_SVG,
    DEBUG_DISPLAY_ALL_TEMPERAMENTS,
    DEBUG_RANDOMLY_ASSIGN_FRIENDS, 
    DEBUG_RANDOMLY_ASSIGN_FRIENDS_TOTAL,
    DEBUG_EVERY_INTERACTION_IS_FRIENDSHIP,
    DEBUG_EVERY_INTERACTION_IS_A_BITE,
    NFC_COLLECTION,
    NFC_SET,
    AnimalInteractions,
    AnimalInteractionMessages,
    ClickTargets,
    ClickTargetMessages
} from './GameConstants.js';

import {
    animalNames
} from './AnimalNames.js';

export class Animals {

    totalPlayers = null;

    // a reference to our NFC class
    nfc = null;

    //
    useNFC = null;

    bites = document.querySelectorAll('.bite');
    biteContainer = document.getElementById('biteContainer');
    
    currentBites = 0;
    
    // values imported from config
    animalsTotalSnarlish = null;
    maxBites = null;

    randomSortOrder = [];

    animals = {
        1: {
            name: "Marshmallow",
            type: "bunny",
            visible: true,
            nfc: null,
            pronoun: "her",
            temperament: [],  // was temperament
            snarlish: true,
            startingNumber: null,
            friendOf: null,
            image: "bunny.png",
            gestures: {
                friendly: [
                    { msg: "{name} playfully snatches the treat and crunches away in delight.", count: 0 },
                    { msg: "With a flick of {pronoun} ears and a joyful hop, {name} shows {pronoun} excitement.", count: 0 },
                    { msg: "{name} finishes {pronoun} treat and gently nudges your hand with {pronoun} nose.", count: 0 },
                    { msg: "After savoring the treat, {name} flops onto {pronoun} side, utterly content.", count: 0 }
                ],
                unfriendly: [
                    { msg: "With an indignant flick of {pronoun} ears, {name} turns away, leaving the treat untouched.", count: 0 },
                    { msg: "Scrunching {pronoun} nose, {name} sniffs the treat briefly before hopping off in silent protest.", count: 0 },
                    { msg: "A sharp thump of {name}'s hind leg echoes, a clear declaration of displeasure.", count: 0 },
                    { msg: "With a dramatic toss of its head, {name} refuses the treat and sulks in the corner.", count: 0 },
                    { msg: "{name} narrows {pronoun} eyes, nudges the treat aside, and looks at you with a disapproving glare.", count: 0 }
                ]
            }
        },

        2: {
            name: "Willow",
            type: "cat",
            visible: true, 
            nfc: null,                       
            pronoun: "her",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "cat.png",
            gestures: {
                friendly: [
                    { msg: "{name} purrs contentedly while nibbling the treat, {pronoun} eyes half-closed in bliss.", count: 0 },
                    { msg: "With graceful precision, {name} gently takes the treat from your fingers and settles down to savor it.", count: 0 },
                    { msg: "{name} headbutts your hand affectionately after finishing {pronoun} treat.", count: 0 },
                    { msg: "A happy trill escapes {name} as {pronoun} delicately accepts the offering.", count: 0 },
                    { msg: "{name} wraps {pronoun} tail around your leg while enjoying the treat, showing {pronoun} appreciation.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} flicks {pronoun} tail dismissively and turns {pronoun} back to the treat.", count: 0 },
                    { msg: "With a haughty sniff, {name} walks away from the treat, {pronoun} tail held high.", count: 0 },
                    { msg: "{name} narrows {pronoun} eyes and lets out a small hiss of disapproval.", count: 0 },
                    { msg: "Ears flattened, {name} swats the treat away with {pronoun} paw.", count: 0 },
                    { msg: "{name} looks at the treat, then at you, then deliberately knocks it off the surface with {pronoun} paw.", count: 0 }
                ]
            }            
        },

        3: {

            name: "Athena",
            type: "owl",
            visible: true,  
            nfc: null,                      
            pronoun: "her",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "owl.png",
            gestures: {
                friendly: [
                    { msg: "{name} swivels {pronoun} head gracefully and delicately accepts the treat with {pronoun} beak.", count: 0 },
                    { msg: "With a soft 'hoo', {name} gently takes the treat, {pronoun} feathers puffing up contentedly.", count: 0 },
                    { msg: "{name} bobs {pronoun} head in appreciation before carefully picking up the treat.", count: 0 },
                    { msg: "Blinking {pronoun} large eyes slowly, {name} shows trust while accepting the offering.", count: 0 },
                    { msg: "{name} spreads {pronoun} wings slightly in delight as {pronoun} enjoys the treat.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} rotates {pronoun} head 180 degrees away from the treat in clear rejection.", count: 0 },
                    { msg: "With a sharp clack of {pronoun} beak, {name} demonstrates {pronoun} displeasure.", count: 0 },
                    { msg: "{name} ruffles {pronoun} feathers and maintains an intimidating stare.", count: 0 },
                    { msg: "Making {pronoun} body appear larger, {name} puffs up {pronoun} feathers defensively.", count: 0 },
                    { msg: "{name} lets out a low warning screech, {pronoun} talons gripping the perch tightly.", count: 0 }
                ]
            }
        },


        4: {
            name: "Waddles",
            type: "penguin",
            visible: true,    
            nfc: null,                   
            pronoun: "his",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "penguin.png",
            gestures: {
                friendly: [
                    { msg: "{name} waddles excitedly and catches the treat with perfect timing.", count: 0 },
                    { msg: "With a happy flipper flap, {name} gracefully accepts the treat.", count: 0 },
                    { msg: "{name} slides on {pronoun} belly towards the treat with enthusiasm.", count: 0 },
                    { msg: "Bobbing {pronoun} head joyfully, {name} picks up the treat with {pronoun} beak.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} turns {pronoun} back and toboggins away from the treat.", count: 0 },
                    { msg: "With a grumpy squawk, {name} ignores the treat entirely.", count: 0 },
                    { msg: "{name} shakes {pronoun} head and waddles away in protest.", count: 0 },
                    { msg: "Flippers crossed, {name} looks pointedly away from the offering.", count: 0 }
                ]
            }
        },

        5: {
            name: "Grizzly",
            type: "bear",
            visible: true,
            nfc: null,                        
            pronoun: "his",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "bear.png",
            gestures: {
                friendly: [
                    { msg: "{name} gently takes the treat with {pronoun} massive paw, showing surprising delicacy.", count: 0 },
                    { msg: "Sitting back on {pronoun} haunches, {name} happily munches the treat.", count: 0 },
                    { msg: "{name} lets out a content rumble while accepting the offering.", count: 0 },
                    { msg: "With surprising gentleness, {name} carefully picks up the treat with {pronoun} claws.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} lets out a disapproving huff and turns {pronoun} broad shoulders away.", count: 0 },
                    { msg: "Standing to {pronoun} full height, {name} shows {pronoun} disinterest dramatically.", count: 0 },
                    { msg: "{name} grumbles deeply and ambles away from the treat.", count: 0 },
                    { msg: "With a mighty yawn, {name} shows {pronoun} complete lack of interest.", count: 0 }
                ]
            }
        },
        
        6: {
            name: "Scout",
            type: "puppy",
            visible: true, 
            nfc: null,                       
            pronoun: "his",            
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "puppy.png",
            gestures: {
                friendly: [
                    { msg: "{name} bounces excitedly, tail wagging at supersonic speeds.", count: 0 },
                    { msg: "With puppy enthusiasm, {name} performs a perfect sit before taking the treat.", count: 0 },
                    { msg: "{name} gives happy little yips before gently taking the treat.", count: 0 },
                    { msg: "Tail creating a windstorm, {name} carefully takes the treat with {pronoun} tiny teeth.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} flops dramatically onto {pronoun} side, ignoring the treat entirely.", count: 0 },
                    { msg: "With a puppy pout, {name} turns {pronoun} head away from the treat.", count: 0 },
                    { msg: "{name} lets out a disappointed little whimper and backs away.", count: 0 },
                    { msg: "Giving {pronoun} best sad puppy eyes, {name} refuses to acknowledge the treat.", count: 0 }
                ]
            }
        },
        7: {
            name: "Eucalyptus",
            type: "koala",
            visible: true,    
            nfc: null,                    
            pronoun: "his",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "koala.png",
            gestures: {
                friendly: [
                    { msg: "{name} slowly blinks and reaches for the treat with {pronoun} fuzzy paw.", count: 0 },
                    { msg: "With sleepy contentment, {name} accepts the treat while hugging {pronoun} branch.", count: 0 },
                    { msg: "{name} lazily shifts position to better reach the offering.", count: 0 },
                    { msg: "Moving with deliberate slowness, {name} gratefully accepts the treat.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} continues {pronoun} nap, completely ignoring the treat.", count: 0 },
                    { msg: "Without opening {pronoun} eyes, {name} turns away from the offering.", count: 0 },
                    { msg: "{name} hugs {pronoun} tree tighter and pretends not to notice.", count: 0 },
                    { msg: "With a sleepy grunt, {name} shows {pronoun} complete lack of interest.", count: 0 }
                ]
            }
        },

        8: {
            name: "Peep",
            type: "baby chick",
            visible: true,   
            nfc: null,                     
            pronoun: "its",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "chick.png",
            gestures: {
                friendly: [
                    { msg: "{name} chirps excitedly and hops toward the treat with tiny steps.", count: 0 },
                    { msg: "With adorable determination, {name} pecks at the treat precisely.", count: 0 },
                    { msg: "{name} fluffs up {pronoun} downy feathers in delight.", count: 0 },
                    { msg: "Cheeping happily, {name} accepts the treat with {pronoun} tiny beak.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} lets out an indignant peep and toddles away.", count: 0 },
                    { msg: "With wobbly steps, {name} retreats from the offering.", count: 0 },
                    { msg: "{name} hides behind {pronoun} own tiny wings.", count: 0 },
                    { msg: "Fluffing up in protest, {name} turns {pronoun} tiny tail feathers to the treat.", count: 0 }
                ]
            }
        },
        
        9: {
            name: "Grace",
            type: "swan",
            visible: true,    
            nfc: null,                    
            pronoun: "her",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "swan.png",
            gestures: {
                friendly: [
                    { msg: "{name} glides forward elegantly to accept the offering.", count: 0 },
                    { msg: "With royal grace, {name} extends {pronoun} long neck to take the treat.", count: 0 },
                    { msg: "{name} curves {pronoun} neck into a heart shape before accepting.", count: 0 },
                    { msg: "Dipping {pronoun} head regally, {name} receives the treat with dignity.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} raises {pronoun} wings slightly in aristocratic disapproval.", count: 0 },
                    { msg: "With regal disdain, {name} swims away from the offering.", count: 0 },
                    { msg: "{name} holds {pronoun} head high and ignores the treat completely.", count: 0 },
                    { msg: "Making a soft hiss, {name} shows {pronoun} royal disapproval.", count: 0 }
                ]
            }
        },
        
        10: {
            name: "Spike",
            type: "porcupine",
            visible: true,   
            nfc: null,                     
            pronoun: "his",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "porcupine.png",
            gestures: {
                friendly: [
                    { msg: "{name} carefully lowers {pronoun} quills to accept the treat safely.", count: 0 },
                    { msg: "With gentle movements, {name} reaches for the treat with {pronoun} paws.", count: 0 },
                    { msg: "{name} chatters happily while taking the offering.", count: 0 },
                    { msg: "Nose twitching with interest, {name} delicately accepts the treat.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} raises {pronoun} quills slightly in warning.", count: 0 },
                    { msg: "With a rattling of quills, {name} turns away from the treat.", count: 0 },
                    { msg: "{name} puffs up {pronoun} spikes and backs away slowly.", count: 0 },
                    { msg: "Making a soft clacking sound, {name} shows {pronoun} displeasure.", count: 0 }
                ]
            }
        },
        
        11:{
            name: "Nutkin",
            type: "squirrel",
            visible: true,  
            nfc: null,                      
            pronoun: "his",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "squirrel.png",
            gestures: {
                friendly: [
                    { msg: "{name} scampers forward eagerly, {pronoun} tail twitching with excitement.", count: 0 },
                    { msg: "Standing on {pronoun} hind legs, {name} accepts the treat with tiny paws.", count: 0 },
                    { msg: "{name} chatters gratefully while stashing the treat in {pronoun} cheeks.", count: 0 },
                    { msg: "With quick, precise movements, {name} examines and accepts the offering.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} flicks {pronoun} tail in agitation and backs away.", count: 0 },
                    { msg: "With a warning chitter, {name} darts away from the treat.", count: 0 },
                    { msg: "{name} stamps {pronoun} tiny feet in protest.", count: 0 },
                    { msg: "Letting out an alarm call, {name} shows {pronoun} disapproval.", count: 0 }
                ]
            }
        },
        
        12: {
            name: "Splash",
            type: "seal",
            visible: true,   
            nfc: null,                     
            pronoun: "her",
            temperament: [],
            snarlish: false,
            startingNumber: null,
            friendOf: null,
            image: "seal.png",
            gestures: {
                friendly: [
                    { msg: "{name} claps {pronoun} flippers together joyfully before taking the treat.", count: 0 },
                    { msg: "With a happy bark, {name} catches the treat expertly.", count: 0 },
                    { msg: "{name} does a little spin in the water before accepting the offering.", count: 0 },
                    { msg: "Wiggling {pronoun} whiskers cheerfully, {name} takes the treat.", count: 0 }
                ],
                unfriendly: [
                    { msg: "{name} dives underwater, leaving the treat behind.", count: 0 },
                    { msg: "With a disapproving 'ort-ort', {name} turns {pronoun} back.", count: 0 },
                    { msg: "{name} slaps the water with {pronoun} flipper in protest.", count: 0 },
                    { msg: "Rolling away, {name} shows {pronoun} complete lack of interest.", count: 0 }
                ]
            }
        }  
/*            
        */
    };

    constructor(parms) {
        
        this.totalPlayers = parms.totalPlayers;
        this.nfc = parms.nfc;
        this.useNFC = parms.useNFC;

        // import defaults from the config object
        this.maxBites = window.config.maxBites;
        this.animalsTotalSnarlish = window.config.animalsTotalSnarlish;

        // match NFC tags with our collection
        this.matchNFCwithAnimalCollection();

        const totalAnimals = Object.keys(this.animals).length;
        const animalLimit = window.config.animalsLimitTotalVisible;
        console.log(`totalAnimals = ${totalAnimals}`);

        // if our animal limit is less than the total number
        if(animalLimit < totalAnimals) {

            // call our animal limiting function
            this.limitVisibleAnimals(animalLimit);

            console.log(`animal array after limiting`,this.animals);

        }

        // prepare the bite indicator
        this.prepareBiteIndicator();

        // randomly assign new names for all the animals
        this.renameAnimals();

        // create a random sort order
        this.randomSortOrder = this.getAnimalShuffleIndex(this.animals);
        console.log('randomSortOrder = ',this.randomSortOrder);

        // randomly assign animal temperaments
        this.assignTemperaments();

        // if defined, randomly assign some animals to players
        if(DEBUG_RANDOMLY_ASSIGN_FRIENDS) {
            this.debugRandomlyAssignFriends(DEBUG_RANDOMLY_ASSIGN_FRIENDS_TOTAL, this.totalPlayers);
        }
        
    } /* end contructor() */

    matchNFCwithAnimalCollection() {

        // match our collection with defined NFC tags
        const matchedNFC = this.nfc.getCollection(NFC_COLLECTION, NFC_SET);
        // matchedNFC = ["bunny", "puppy", "swan", "cat"];
        console.log('matchedNFC = ',matchedNFC);

        // iterate through each animal in this.animals
        for (const id in this.animals) {
            const animal = this.animals[id];

            // Mark nfc as true if animal type is in matchedNFC array, false otherwise
            animal.nfc = matchedNFC.includes(animal.type);
        }

        console.log('animals after matching = ',this.animals);
    
    }

    // to assist with debugging, we'll limit the display of animals to
    // a subset of all the available animals for non-NFC game play
    limitVisibleAnimals(limitTo) {
        
        if (!Number.isInteger(limitTo) || limitTo < 0) {
            throw new Error('Limit must be a non-negative integer');
        }

        // Convert to array for easier manipulation
        const animalEntries = Object.entries(this.animals);
        const totalAnimals = animalEntries.length;

        // If limit is greater than total, no changes needed
        if (limitTo >= totalAnimals) {
            return;
        }

        // Calculate how many need to be hidden
        const numToHide = totalAnimals - limitTo;
        
        // Create array of indices and shuffle it
        const indices = Array.from({ length: totalAnimals }, (_, i) => i);
        for (let i = indices.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [indices[i], indices[j]] = [indices[j], indices[i]];
        }

        // Set visible: false for randomly selected entries
        indices.slice(0, numToHide).forEach(idx => {
            const key = animalEntries[idx][0];
            this.animals[key] = { ...this.animals[key], visible: false };
        });

    }

    debugRandomlyAssignFriends(numberToAssign, totalPlayers) {

        // Validate inputs
        if (numberToAssign > Object.keys(this.animals).length) {
            throw new Error("Cannot assign more animals than available");
        }
        
        // Create array of available animal IDs
        const availableAnimals = Object.keys(this.animals)
            .filter(id => this.animals[id].friendOf === null)
            .map(Number);

        // Check if we have enough unassigned animals
        if (numberToAssign > availableAnimals.length) {
            throw new Error("Not enough unassigned animals available");
        }
        
        // Randomly assign animals
        for (let i = 0; i < numberToAssign; i++) {
            // Get random index from available animals
            const randomIndex = Math.floor(Math.random() * availableAnimals.length);
            const animalId = availableAnimals[randomIndex];
            
            // Randomly select a player (1 to totalPlayers)
            const randomPlayer = Math.floor(Math.random() * totalPlayers) + 1;
            
            // Assign the animal to the player
            this.animals[animalId].friendOf = randomPlayer;
            console.log(`debugRandomlyAssignFriends() assigned ${animalId} to ${randomPlayer}`)
            
            // Remove the assigned animal from available array
            availableAnimals.splice(randomIndex, 1);
        }
        
        console.log('debugRandomlyAssignFriends() animals after random assignment: ',this.animals)

        return;
    }


    // Function to generate random sort order
    getAnimalShuffleIndex(obj) {

        const keys = Object.keys(obj);
        const randomOrder = keys.map((_, index) => index);
        
        for (let i = randomOrder.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [randomOrder[i], randomOrder[j]] = [randomOrder[j], randomOrder[i]];
        }
        
        return keys.reduce((acc, key, index) => {
            acc[key] = randomOrder[index];
            return acc;
        }, {});

    }

    // Function to get items in random order without modifying original
    // generated by Claude
    // NOT sure it's working so we're not using it
    getRandomSortedItems(obj, sortOrder) {

        const entries = Object.entries(obj);
    
        // First, create array with sort indices
        const sorted = entries
            .map(entry => ({
                key: entry[0],
                value: entry[1],
                sortIndex: sortOrder[entry[0]]
            }))
            .sort((a, b) => a.sortIndex - b.sortIndex);
        
        // Create new object with sequential keys (1,2,3,4...)
        return sorted.reduce((acc, item, index) => {
            acc[index + 1] = item.value;
            return acc;
        }, {});

    }

    createHeart() {
        const heart = document.createElement('div');
        heart.className = 'heart';
        heart.textContent = '❤️';
        heart.style.zIndex = '-1';
        
        // Random starting position
        const startX = Math.random() * window.innerWidth;
        heart.style.left = startX + 'px';
        heart.style.bottom = '0px';
        
        // Random horizontal drift
        const offsetX = (Math.random() - 0.5) * 200;
        heart.style.setProperty('--offsetX', offsetX + 'px');
        
        document.body.appendChild(heart);
        
        // Remove heart after animation completes
        heart.addEventListener('animationend', () => {
          heart.remove();
        });
    }

    createFloatingImage() {

        const config = {
            //imageUrl: 'assets/img/angry-emoji_w30px.png',
            //imageUrl: 'assets/img/fangs_w75px.png',
            //imageUrl: 'assets/img/tiger-teeth-icon_w75px.png',
            imageUrl: 'assets/img/bare-tiger-teeth_w75px.png',
            imageSize: 40,
            spawnInterval: 200,
            spreadDistance: 200
        };

        const container = document.createElement('div');
        container.className = 'floating-bite';
        container.style.zIndex = '-1';
        
        const img = document.createElement('img');
        img.src = config.imageUrl;
        img.alt = 'Floating';
        container.appendChild(img);
        
        // Get the source container's position
        const sourceContainer = document.querySelector('.animal-card');
        //const sourceContainer = document.getElementById("playerTreats");
        const rect = sourceContainer.getBoundingClientRect();
        const centerX = rect.left + rect.width / 2 - config.imageSize / 2;
        const centerY = rect.top + rect.height / 2 - config.imageSize / 2;
        
        // Set starting position at the center
        container.style.left = centerX + 'px';
        container.style.top = centerY + 'px';
        
        // Calculate random angle for spread direction
        const angle = Math.random() * Math.PI * 2;
        const distance = config.spreadDistance;
        
        // Calculate offset based on angle
        const offsetX = Math.cos(angle) * distance;
        const offsetY = Math.sin(angle) * distance;
        const rotation = Math.random() * 360;
        
        container.style.setProperty('--offsetX', offsetX + 'px');
        container.style.setProperty('--offsetY', offsetY + 'px');
        container.style.setProperty('--rotation', rotation);
        
        document.body.appendChild(container);
        
        container.addEventListener('animationend', () => {
            container.remove();
        });
    }

    // returns a prepared animalID
    // otherwise, returns null if invalid
    validateAnimalID(animalID) {

        console.log(`validateAnimalID() animalID = ${animalID}`);

        // Convert to number if it's a string number
        const id = Number(animalID);

/*        
        if (isNaN(animalID)) {
            throw new Error('Animal ID must be a number');
        }
*/
        // Check if ID is positive
        if (id <= 0) {
            throw new Error('Animal ID must be positive');
        }
    
        // Check if animalID exists in the animals object
        if (!this.animals.hasOwnProperty(id)) {
            throw new Error(`Invalid animal ID: ${id}. Animal does not exist.`);
        }

        console.log(`validateAnimalID() valid prepared id = ${id}`);

        return id;

    }


    getAnimalName(animalID) {

        const validatedAnimalID = this.validateAnimalID(animalID);

        if(!validatedAnimalID) {

            return null;

        } else {
    
            console.log(`getAnimalName() validatedAnimalID = ${validatedAnimalID}`);

            const animalName = this.animals[validatedAnimalID].name;

            return animalName;

        }

    }

    getAnimalPronoun(animalID, possessivePronoun = false) {

        const validatedAnimalID = this.validateAnimalID(animalID);
    
        if(!validatedAnimalID) {

            return null;

        } else {

            if(possessivePronoun) {

                const pronounMap = {
                    'his': 'him',
                    'her': 'her',  // stays the same
                    'its': 'them'
                };
    
                const storedPronoun = this.animals[animalID].pronoun;
                return pronounMap[storedPronoun] || storedPronoun; // fallback to original if     
    
            } else {
    
                // just return what's stored for this animal
                const pronoun = this.animals[id].pronoun;
                return pronoun;
        
            }

        }
    
    }

    totalAnimals() {
    
        return Object.keys(this.animals).length;

    }

    getAnimal(input) {

        console.log("getAnimal() called with input = ",input);

        if (typeof input === 'number' || parseInt(input) > 0) {
            
            // If input is a number, return the animal with that id
            //return this.animals[1];
            const id = parseInt(input);

            //console.log("getAnimal() showing the collection: ",this.animals);

            return this.animals[id];
            //return Object.values(this.animals)[id];

        } else if (typeof input === 'string') {
            
            // If input is a string, filter animals by type
            const animalsMatched = Object.values(this.animals).filter(animal => animal.type === input);

            return animalsMatched[0];

        }

        console.log("getAnimal() couldn't match the type of input = ",typeof input);
        return null; // Return null for invalid inputs        

    }

    // when provided an animalType from an NFC scan,
    // return the matching animalID
    matchAnimalTypeToID(animalType) {

        // Using Object.entries()
        //const animalID = Object.entries(animals).find(([key, animal]) => animal.type === "bunny")[0];

        // Or using Object.keys()
        const animalID = Object.keys(this.animals).find(key => this.animals[key].type === animalType);

        return animalID;

    }

    assignSnarlish(animalsTotalSnarlish) {
        // If animalsTotalSnarlish is null, return without changes
        if (animalsTotalSnarlish === null) {
            return;
        }
      
        // if using NFC, get those marked as NFC = true,
        // otherwise pick the ones marked visible = true
        const eligibleAnimalKeys = this.useNFC 
            ? Object.keys(this.animals).filter(key => this.animals[key].nfc === true)
            : Object.keys(this.animals).filter(key => this.animals[key].visible === true);
        
        const totalEligibleAnimals = eligibleAnimalKeys.length;
      
        // Validate input
        if (animalsTotalSnarlish < 0) {
            throw new Error('animalsTotalSnarlish must be non-negative');
        }
      
        // Ensure we don't try to assign more than total eligible animals
        const numToAssign = Math.min(animalsTotalSnarlish, totalEligibleAnimals);
      
        // Reset all animals to not snarlish
        Object.keys(this.animals).forEach(key => {
            this.animals[key].snarlish = false;
        });
      
        // Randomly select eligible animals to be snarlish
        const selectedIndices = new Set();
        while (selectedIndices.size < numToAssign) {
            const randomIndex = Math.floor(Math.random() * totalEligibleAnimals);
            if (!selectedIndices.has(randomIndex)) {
                selectedIndices.add(randomIndex);
                const key = eligibleAnimalKeys[randomIndex];
                this.animals[key].snarlish = true;
            }
        }

        // Return array of animals marked as snarlish
        return Object.keys(this.animals)
            .filter(key => this.animals[key].snarlish)
            .map(key => ({ ...this.animals[key], id: key }));
    }
    
    assignTemperaments() {

        // before we begin, select our snarlish animals based on animalsTotalSnarlish
        const snarlishAnimals = this.assignSnarlish(this.animalsTotalSnarlish);
        console.log('snarlish animals = ',snarlishAnimals);

        Object.entries(this.animals).forEach(([key, animal]) => {

            let temperament = [];

            // once per player
            for (let i = 0; i < this.totalPlayers; i++) {

                let temperamentFloat = 0;

                // handle snarlish animals
                if(animal.snarlish) {
                
                    // snarlish animals will be at the min
                    temperamentFloat = -1;

                } else {

                    const playerTemperament = (Math.random() * 1.6 - 0.8);
                    //console.log(playerTemperament);
    
                    temperamentFloat = parseFloat(playerTemperament.toFixed(2));

                }
            
                temperament.push(temperamentFloat);
            }

            //temperament = [-.5,-.2];

            // save this value to our collection
            this.animals[key].temperament = temperament;

            console.log(`${animal.type} temperament `,temperament);

        });

    }

    getRandomGesture(id, type = 'friendly') {
        const animal = this.getAnimal(id);
        if (!animal) return null;

        const gestures = animal.gestures[type];
        
        // Find the minimum count among all gestures
        const minCount = Math.min(...gestures.map(g => g.count));
        
        // Filter gestures to only include those with the minimum count
        const leastUsedGestures = gestures.filter(g => g.count === minCount);
        
        // Randomly select from the least used gestures
        const randomIndex = Math.floor(Math.random() * leastUsedGestures.length);
        const selectedGesture = leastUsedGestures[randomIndex];
        
        // Find the original index in the gestures array to update the count
        const originalIndex = gestures.findIndex(g => g.msg === selectedGesture.msg);
        gestures[originalIndex].count++;
        
        // Replace both {name} and {pronoun} placeholders
        return selectedGesture.msg
            .replace("{name}", animal.name)
            .replace(/{pronoun}/g, animal.pronoun);
    }

    addAnimal(animal) {
        const id = Math.max(...Object.keys(this.animals).map(Number)) + 1;
        this.animals[id] = { ...animal };
        return id;
    }

    updatetemperament(id, newTemperament) {
        const animal = this.getAnimal(id);
        if (animal) {
            animal.temperament = newTemperament;
            return true;
        }
        return false;
    }

    setFriendOf(id, friendId) {
        const animal = this.getAnimal(id);
        if (animal) {
            animal.friendOf = friendId;
            return true;
        }
        return false;
    }

    getGestureStats(id) {
        const animal = this.getAnimal(id);
        if (!animal) return null;

        return {
            friendly: animal.gestures.friendly.map(g => ({ message: g.msg, count: g.count })),
            unfriendly: animal.gestures.unfriendly.map(g => ({ message: g.msg, count: g.count }))
        };
    }
   
    renameAnimals() {

        // animalNames
        // was: animalObj, namesData
        
        Object.entries(this.animals).forEach(([key, animal]) => {

            const animalType = animal.type;
            const availableNames = animalNames[animalType];

            if (!availableNames) {
                
                console.log(`No names available for animal type: ${animalType}`);

            } else {

                const randomNameData = availableNames[Math.floor(Math.random() * availableNames.length)];

                const newName = randomNameData.name;
                const newPronoun = randomNameData.pronoun ?? (Math.random() < 0.5 ? 'her' : 'his');

                //console.log(`${animalType} is now named ${newName} with pronoun = ${newPronoun}`);

                // update the name and pronoun in our collection
                this.animals[key].name = newName;
                this.animals[key].pronoun = newPronoun;

            }

        });

    } /* end renameAnimals() */


    /*******************************************************************************
     * @function handleAnimalInteraction
     *
     * @description Handles any animal interaction that has taken place as a
     * result of feeding the animal a treat.
     * 
     * @details
     * - 
	 * - 
	 * - 
     * 
     * @param {integer} animalID - an index of one of the animals
     * @param {object} currentPlayerObj - the current player object
     * @param {object} treatObj - the treat being fed,
     *    may be null if this is a game opening interaction
     * @returns {integer} interactionResult - the results of the interaction
     *    or null if the interaction was out of turn
     * 
     * @example handleAnimalInteraction(animalID, currentPlayerObj);
	 * 
	 * @see 
     * 
     * @author Kristi
     * @since 2025-01-05
     * @lastModified 2025-01-05
     ******************************************************************************/

    handleAnimalInteraction(animalID, currentPlayerObj, treatObj = null) {

        // Examples:
        // animalID = integer 1 through 12 (or more)
        // currentPlayerObj = { number: 1, color: null, animals: 0, bites: 1, turns: 0, name: "Kristi" }
        // treatObj = {value: 60, name: "Nachos"}

        console.log(`handleAnimalInteraction() handling animal interaction`);

        // check if this animal already has a friend
        const currentFriend = this.animals[animalID].friendOf;
        if(currentFriend) {

            // this animal already has a friend

            // player trying to re-interact with one of their animals

            // player trying to interaction with someone else's animal

            return AnimalInteractions.Nothing;

        }

        if(DEBUG_EVERY_INTERACTION_IS_FRIENDSHIP && DEBUG_EVERY_INTERACTION_IS_A_BITE) {

            // both are set as options, so randomly choose one

            const isInteractionFriendship = Math.random() < 0.5;
            if (isInteractionFriendship) {
                // Handle friendship interaction
                return AnimalInteractions.NewFriend;
            } else {
                // Handle bite interaction
                return AnimalInteractions.NewBite;
            }

        }

        if(DEBUG_EVERY_INTERACTION_IS_FRIENDSHIP) {

            return AnimalInteractions.NewFriend;

        }

        if(DEBUG_EVERY_INTERACTION_IS_A_BITE) {
            
            return AnimalInteractions.NewBite;

        }

        // get the value of the specified treat
        const treatValue = treatObj[0] && treatObj[0].value;
        console.log(`handleAnimalInteraction() treatValue = ${treatValue}`);

        // does the treat have a value?
        if(!treatValue) {

            // no treat specified or it's value is zero
            // no interaction
    
            return AnimalInteractions.Nothing;

        }

        const result = this.feedTreat(animalID, currentPlayerObj.number, treatValue);


        // default interaction to return
        return result;
	}


    // given an animalID and a treatValue, 
    // calculate how this affects temperament
    feedTreat(animalID, playerNumber, treatValue) {

        const playerID = playerNumber - 1;

        console.log(`feedTreat() animalID = ${animalID}`);
        console.log(`feedTreat() playerID = ${playerID}`);
        console.log(`treatValue() treatValue = ${treatValue}`);        

        // if this animal is snarlish, nothing will change it's temperament and it's a guarantee bite
        const isSnarlish = this.animals[animalID].snarlish;
        if(isSnarlish) {

            console.log(`feedTreat() this animal is snarlish`);

            return AnimalInteractions.NewBite;

        }

        // get this animal's current temperament towards this player
        const temperamentTowardsPlayer = this.animals[animalID].temperament[playerID];
        console.log(`feedTreat() temperamentTowardsPlayer = ${temperamentTowardsPlayer}`);

        const result = this.handleFeeding(temperamentTowardsPlayer, treatValue);
        console.log(`feedTreat() feeding result = `,result);

        /*

        sample result:
            {
                satisfaction: 0.4614285714285714,
                postTemperament: -0.018571428571428572,
                biteChance: 0.018571428571428572,
                willBite: false
            }
        */
        
        // update the animal's temperament for this player
        this.animals[animalID].temperament[playerID] = result.postTemperament;

        // handle a new friend
        if(result.postTemperament == 1) {

            // mark this animal as a friend
            this.animals[animalID].friendOf = playerID;
            console.log(this.animals);

            return AnimalInteractions.NewFriend;
            
        }

        // handle the bite
        if(result.willBite == true) {

            return AnimalInteractions.NewBite;
            
        }

        // return the default
        return AnimalInteractions.Nothing;


    }

    // https://claude.ai/chat/1e5bb01e-bae4-428a-8f79-d2f721b8cbfa

    calculateHungerSatisfaction(treatValue) {
        const normalizedTreat = (treatValue - 5) / (75 - 5);
        return normalizedTreat * 0.75; // Increased from 0.5 to 0.75
    }

    calculateHungerSatisfaction(temperament, treatValue) {
        // Normalize treat weight to 0-1 range
        const normalizedTreat = (treatValue - 5) / (75 - 5);
        
        // linear satisfaction based only on treat weight
        
        // Maximum treat (75) takes 2.0 turns
        // Medium treat (40) takes 4.0 turns
        // Small treat (15) takes 14.0 turns
        //return normalizedTreat * 0.5;

        // Maximum treat (75) takes 1.3 turns
        // Medium treat (40) takes 2.7 turns
        // Small treat (15) takes 9.3 turns
        return normalizedTreat * 0.75;
    }
    
    calculateBiteProbability(postFeedTemperament) {
        if (postFeedTemperament >= 0) return 0;
        return Math.abs(postFeedTemperament);
    }

/* mapping treat values to temperment values (V1)    
    calculateHungerSatisfaction(temperament, treatValue) {
        // Normalize treat weight to 0-1 range
        const normalizedTreat = (treatValue - 5) / (75 - 5);
        
        // Base satisfaction for neutral temperament (0)
        const baseSatisfaction = normalizedTreat * 0.5;
        
        // Temperament modifier: converts -1 to 1 range to 0 to 1
        const temperamentModifier = (temperament + 1) / 2;
        
        return baseSatisfaction * (0.5 + temperamentModifier);
    }

    calculateBiteProbability(postFeedTemperament) {
        // Only bite chance when temperament is below 0
        if (postFeedTemperament >= 0) return 0;
        
        // Linear scaling from 0% at 0 to 100% at -1
        return Math.abs(postFeedTemperament);
    }
*/    

    // Combined feeding and bite check function
    handleFeeding(preTemperament, treatValue) {
        const satisfaction = this.calculateHungerSatisfaction(preTemperament, treatValue);
        
        // postTemperament is preTemperament + satisfaction, but no greater than 1
        const postTemperament = Math.min(1, preTemperament + satisfaction); // Replace with your temperament change logic
        
        const biteChance = this.calculateBiteProbability(postTemperament);
        const willBite = Math.random() < biteChance;
        
        return {
            satisfaction,
            postTemperament,
            willBite,
            biteChance
        };
    }

    /*******************************************************************************
	function prepareAnimalCard()
	
	Prepares a DOM object for the specified animal
    
		
	@param  animalID        the ID of the animal to prepare   
    @param  outerWrapper    boolean whether or not to include the outer wrapper
                            (optional = false)  
    
	Returns: a DOM object for the specified animal 
             or null if the animalID didn't match

    ******************************************************************************/	    

    prepareAnimalCard(parms) {

        const animalID = parms.animalID;
        const outerWrapper = parms.outerWrapper ?? false;
        const displayMany = parms.displayMany ?? false;
        const displayStats = parms.displayStats ?? false;
        const interaction = parms.interaction ?? null;
        const target = parms.target ?? ClickTargets.NextPlayer;

        // validate that the animalID matches an item in our collection
        const animal = this.getAnimal(animalID);

        console.log("prepareAnimalCard() animal = ",animal)

        if(!animal) {

            console.log(`prepareAnimalCard() can't find an animal matching animalID = ${animalID}`);
            return(null);

        }

        // we'll define mainContainer here whether or not we'll use it
        const mainContainer = document.createElement('div');
        //const innerContainer = document.createElement('div');

        if(outerWrapper) {

            if(displayMany) {

                // prepare a wrapper for many


            } else {

                // prepare a wrapper for just one
                mainContainer.className = 'animals-container';
                mainContainer.style.cssText = `
                    margin-top: 10px;
                    display: flex;
                    justify-content: center;
                    width: 100%;
                    display: flex;
                    flex-wrap: wrap;
                    gap: 20px;  /* or use margin on the cards themselves */
                `;

            }

            
            //mainContainer.style.cssText = 'display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; padding: 20px;';
/*
            mainContainer.style.cssText = `
                max-width: 1200px;
                margin: 0 auto;
                display: grid; 
                grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
                gap: 20px;
                padding: 20px;
                justify-content: center;
                justify-items: center;
                align-items: center;
            `;
*/

            //mainContainer.style.cssText = 'display: flex; justify-content: center; width: 100%;';
            
            /*
            innerContainer.style.cssText = `
                width: 1200px;
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
                gap: 20px;
                padding: 20px;
            `;
            */

            // Restructure DOM
            //mainContainer.appendChild(innerContainer);
        }

        const animalWrapper = document.createElement('div');
        animalWrapper.className = 'animal-card';
        
        //animalWrapper.style.cssText = 
        `
            flex: 0 1 250px;  /* don't grow, can shrink, base width 300px */
            margin-left: auto;
            margin-right: auto;
            border: 1px solid #ccc; 
            border-radius: 8px;
            padding: 15px;
            text-align: center;
            background-color: white;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        `;

        // player has been bitten by this animal
        if(interaction == AnimalInteractions.NewBite) {
            //animalWrapper.style.background = 'radial-gradient(circle,rgb(247, 227, 77),rgb(185, 185, 90)';  // bitten
            animalWrapper.style.background = 'linear-gradient(to right, rgba(248, 136, 216, 0.5), rgba(180, 90, 185, 0.5)';
        }

        // player has made a new friend
        if(interaction == AnimalInteractions.NewFriend) {
            //animalWrapper.style.background = 'linear-gradient(to right,rgb(246, 216, 217),rgb(250, 247, 165))';
            //animalWrapper.style.background = 'radial-gradient(circle,rgb(246, 216, 217),rgb(254, 206, 206)';
            //animalWrapper.style.background = 'linear-gradient(to right, rgba(255, 0, 0, 0.5), rgba(0, 255, 0, 0.5))';
            animalWrapper.style.background = 'linear-gradient(to right, rgba(0, 128, 170, 0.5), rgba(64, 224, 208, 0.5))';
        }

/*        
        // DEPRECIATED -- see below
        // add click event listener
        animalWrapper.addEventListener("click", (event) => { 
            //event.preventDefault(); // prevent the default link behavior
            //this.handleTreatTap(item);
            animalTapped(animalID);
        });
*/
        // add one time click event listener for animal selection
        if(target == ClickTargets.AnimalSelection) {
            animalWrapper.addEventListener("click", (event) => { window.animalTapped(animalID); }, { once: true });
        }

        // add one time click event listener for next player
        if(target == ClickTargets.NextPlayer) {
            animalWrapper.addEventListener("click", (event) => { window.advancePlayer(); }, { once: true });
        }

        // Image container without border radius
        const imageContainer = document.createElement('div');
        imageContainer.style.cssText = `
            width: 150px;
            height: 150px;
            margin: 0 auto 10px auto;
            position: relative;
        `;

        // Image without border radius
        const animalImage = document.createElement('img');
        animalImage.src = 'assets/animals/' + animal.image || PLACEHOLDER_SVG;
        animalImage.onerror = function() {
            this.src = PLACEHOLDER_SVG;
            };            
        animalImage.alt = `${animal.name} the ${animal.type}`;
        animalImage.style.cssText = `
            width: 1333px;
            height: 1512px;
            position: absolute;
            top: 50%;
            left: 50%;
            margin-top: 6px;
            transform: translate(-50%, -50%) scale(0.11);
            transform-origin: center;
        `;
        imageContainer.appendChild(animalImage);
        animalWrapper.appendChild(imageContainer);

        const animalName = document.createElement('h3');
        animalName.textContent = animal.name;
        animalName.style.cssText = 'margin: 0; padding-top: 10px; color: #333; font-size: 1.2em;';
        animalWrapper.appendChild(animalName);

        const animalType = document.createElement('p');
        animalType.textContent = animal.type.charAt(0).toUpperCase() + animal.type.slice(1);
        animalType.style.cssText = 'margin: 5px 0; color: #666; font-style: italic;';
        animalWrapper.appendChild(animalType);

/*      
        // display the pronoun -- during debugging
        const animalPronoun = document.createElement('p');
        animalPronoun.textContent = animal.pronoun.charAt(0).toUpperCase() + animal.pronoun.slice(1);
        animalPronoun.style.cssText = 'margin: 5px 0; color: #666; font-style: italic;';
        animalWrapper.appendChild(animalPronoun);         
*/

        // add empty div for animal temperament
        // this will be filled in later
        const animalTemperamentGraph = document.createElement('div');
        animalTemperamentGraph.className = 'temperament-container';
        animalTemperamentGraph.id = animal.type + '-temperament';
        animalTemperamentGraph.style.display = 'none';
        animalWrapper.appendChild(animalTemperamentGraph);

        if(displayStats) {

            // [DEBUG] here we intented to show how many treat points were needed to win over the animal based on the current player

            /*
            const animalStats = document.createElement('p');
            animalStats.textContent = "ANIMAL STATS HERE";
            animalStats.style.cssText = 'margin: 5px 0; color: #666; font-style: italic;';
            animalWrapper.appendChild(animalStats);  
            */

        }

        animalWrapper.addEventListener('mouseenter', () => {
            animalWrapper.style.transform = 'translateY(-5px)';
            animalWrapper.style.transition = 'transform 0.3s ease';
        });

        animalWrapper.addEventListener('mouseleave', () => {
            animalWrapper.style.transform = 'translateY(0)';
        });
        
        if(parms.target == ClickTargets.NextPlayer) {
            const advancePlayerButton = this.advancePlayerButton();
            animalWrapper.appendChild(advancePlayerButton);    
        }

        if(outerWrapper) {

            // return the mainContainer wrapper
            mainContainer.appendChild(animalWrapper);
            return mainContainer;
        
        } else {

            // no mainContainer wrapper, just return the animalWrapper
            return animalWrapper;

        }
        

    } /* end prepareAnimalCard() */

    addAnimalTemperamentGraph(elementName, temperament, currentPlayerIndex = null) {

        const visualizer = new LikeOMeter(elementName);
        
/*
        const temperament = [
            { name: 'Alice', weight: 0.8 },
            { name: 'Kristi', weight: -1 },            
            { name: 'KT', weight: 1 },         
            { name: 'Mikey', weight: 1 },               
            { name: 'Bob', weight: -0.3 },
            { name: 'Sam', weight: 0 },            
            { name: 'Charlie', weight: 0.2 },
            { name: 'Cindy', weight: -0.7 }
        ];
*/
        visualizer.setTemperament(temperament, currentPlayerIndex);

    }

    addAnimalTemperamentGraphByID(animalID, currentPlayerIndex = null) {

        // TODO validate animalID

        const elementName = this.animals[animalID].type + '-temperament';
        const temperament = this.animals[animalID].temperament;

        console.log('elementName = ',elementName);
        console.log('temperament = ',temperament);

        this.addAnimalTemperamentGraph(elementName, temperament, currentPlayerIndex);
    }

    handleAnimalTap(event) {

        alert("animal tap being handled: " + JSON.stringify(event));


    }

    prepareAnimalStats(animalID, interaction = null) {

        const animalCard = this.prepareAnimalCard({
            animalID: animalID,
            outerWrapper: true, 
            displayStats: true,
            interaction: interaction,
            target: ClickTargets.NextPlayer
        });
    
        return(animalCard);

    }

    advancePlayerButton() {

        let button = document.createElement('div');
        button.className = 'advancePlayer';
        button.style.cssText = `
            width: 65%;
            height: 1.3em;
            margin: 0 auto;
            margin-top: 13px;
            background-color:rgb(104, 98, 199);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            user-select: none;
            border-radius: 15px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }
        `;
        button.innerHTML = "Continue";

/*        
        // since we've made the entire animal card clickable, we needed to remove this to prevent skiping a player
        // add click event listener
        button.addEventListener("click", (event) => {
            //event.preventDefault(); // prevent the default link behavior
            advancePlayer();
        });
*/
        return button;
 
    }

    /*******************************************************************************
	function displayAllAnimals()
	
	Appends our collection of animals to the DOM.
		
	@param  none     
    
	Returns: nothing

    ******************************************************************************/	    
    displayAllAnimals(parms) {

        const availableOnly = parms.availableOnly ?? true;
        const randomizeOrder = parms.randomize ?? true;
        const currentPlayerIndex = parms.currentPlayerIndex ?? null;

        const mainContainer = document.createElement('div');
        mainContainer.className = 'animals-container';
        //mainContainer.style.cssText = 'display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; padding: 20px;';
/*
        let animals = [];

        // if randomizeOrder is true, we'll work with a sorted version of the animal collection
        if(randomizeOrder) {

            animals = this.getRandomSortedItems(this.animals,this.randomSortOrder);

        } else {
        
            // no sorting here, just use a copy
            animals = structuredClone(this.animals);
        
        }
*/
        //console.log('displayAllAnimals() sorted animals = ', animals);

        Object.entries(this.randomSortOrder).forEach(([key, sortIndex]) => {

            // sortIndex is zero indexed and we need it to be 1 indexed
            sortIndex = sortIndex + 1;

            console.log(`displayAllAnimals() key = ${key}, sortIndex = ${sortIndex}`);

            // get friendOf value for this animal
            const friendOf = this.animals[sortIndex].friendOf;
            
            // check for the visible flag for this selected animal
            // but force to true when using NFC
            let visible = this.animals[sortIndex].visible;
            if (this.useNFC) {
                visible = true;
            }

            // display this animal card only if...
            //   visible = true (for non-NFC game play)
            //   availableOnly is true
            //   the animal hasn't been assigned a friend
            //
            if(visible && availableOnly && friendOf === null) {

                //console.log("displayAllAnimals() key = ",key);

                const animalCard = this.prepareAnimalCard(
                        {
                            animalID: sortIndex,
                            outerWrapper: false, 
                            displayStats: false,
                            displayMany: true,
                            target: ClickTargets.AnimalSelection
                        }
                    );

                // attach this card to the collection only if not null
                animalCard && mainContainer.appendChild(animalCard);

            }
            
        });

        //document.body.appendChild(mainContainer);
        const animalSelection = document.getElementById("animalSelection");
        animalSelection.appendChild(mainContainer);

        // display all animal temperament graphs if true
        console.log(`always display temperaments = `,window.config.debugAlwaysDisplayTemperaments);
        if(window.config.debugAlwaysDisplayTemperaments == true) {
            Object.entries(this.animals).forEach(([key, animal]) => {
                // add the temperament graph
                const elementName = animal.type + '-temperament';
                console.log('elementName = ', elementName);
                console.log('temperament = ', animal.temperament);
                this.addAnimalTemperamentGraph(elementName, animal.temperament, currentPlayerIndex);
            });
        }

    } /* end displayAllAnimals() */

    // TODO placeholder until we make this show only available animals
    displayAvailableAnimals(currentPlayerIndex = null) {

        this.displayAllAnimals(
            {
				availableOnly: true,
                currentPlayerIndex: currentPlayerIndex
			}
        );

    }

    // within the biteContainer, create a div for each bite
    prepareBiteIndicator() {
        // Create main container for bites
        const bitesDiv = document.createElement('div');
        bitesDiv.className = 'bites';
    
        // Create individual bite elements based on maxBites
        for (let i = 0; i < this.maxBites; i++) {
            const biteDiv = document.createElement('div');
            biteDiv.className = 'bite';
            bitesDiv.appendChild(biteDiv);
        }
    
        // Clear any existing content and append new bites
        this.biteContainer.innerHTML = '';
        this.biteContainer.appendChild(bitesDiv);
    }

    hideBites() {

        this.biteContainer.style.display = 'none';

    }

    showBites() {

        this.biteContainer.style.display = 'block';
    }    

    // returns a count of the number of animals that have
    // neither been friended
    // nor are snarlish (who can't be friended)
    totalAnimalsRemaining() {

        // during NFC game play
        if(this.useNFC) {

            return Object.values(this.animals).filter(animal => 
                animal.friendOf === null && 
                animal.snarlish === false &&
                animal.nfc === true
            ).length;    
            
        } else {

            // non-NFC game play
            return Object.values(this.animals).filter(animal => 
                animal.friendOf === null && 
                animal.snarlish === false &&
                animal.visible === true
            ).length;
    
        }
    
    }

/*    
    addBite() {
        if (this.currentBites < this.maxBites) {
            this.bites[this.currentBites].classList.add('active');
            this.currentBites++;
        }
    }

    resetBites() {
        this.bites.forEach(bite => this.bite.classList.remove('active'));
        this.currentBites = 0;
    }
*/
    setBites(count) {

        this.bites = document.querySelectorAll('.bite');

        // cast as an integer
        count = parseInt(count);

        console.log(`setBites() count = ${count}`);

        this.bites.forEach((bite, index) => {
            
            if(count <= index) {
                bite.classList.remove('active');
            } else {
                bite.classList.add('active');
            }

        });        

    }


}

class LikeOMeter {

    currentPlayerIndex = null;

    constructor(containerId) {
        this.container = document.getElementById(containerId);

        if(this.container) {
            // make the container visible
            this.container.style.display = 'block';
        }

        this.temperament = [];
    }

    setTemperament(temperament, currentPlayerIndex = null) {

        if(currentPlayerIndex !== null) {
            this.currentPlayerIndex = currentPlayerIndex;
        }

        this.temperament = temperament;
        this.render();
    }

    getEmoji(weight) {
        const absWeight = Math.abs(weight);
        if (weight >= 0) {
            if (absWeight == 1) return '❤️';
            if (absWeight > 0.6) return '🥰';
            if (absWeight > 0.3) return '😊';
            return '🙂';
        } else {
            if (absWeight > 0.7) return '😠';
            if (absWeight > 0.3) return '😕';
            if (absWeight == -1) return '😡';
            return '😐';
        }
    }

    getIconSize(weight) {
        const absWeight = Math.abs(weight);
        //return Math.max(1, 1 + absWeight * 0.5);
        return 1.5;
    }

    getIconPosition(weight) {
        // Convert -1 to 1 range to percentage (0 to 100)
        return ((weight + 1) / 2) * 100;
    }

    render() {
        //this.container.innerHTML = '<h2>Animal temperament</h2>';
        if(!this.container) {
            console.log("element not found - can't add temperament graph");
            return;
        }

        this.temperament.forEach((pref, index) => {
            const row = document.createElement('div');
            row.className = 'temperament-row';
            
            // add highlight for the current player
            if (index === this.currentPlayerIndex) {
                row.classList.add('highlighted');
            }

/*
            // Player Number
            const name = document.createElement('span');
            name.className = 'temperament-player-name';
            //name.textContent = pref.name;
            //name.textContent = 'Player ' + (index+1);
            name.textContent = (index+1);
            row.appendChild(name);
*/
            // Preference scale
            const scale = document.createElement('div');
            scale.className = 'temperament-scale';
/*
            // Add scale markers
            ['Dislike', '|', 'Like'].forEach((text, index) => {
                const marker = document.createElement('span');
                marker.className = `temperament-scale-marker ${index === 0 ? 'left' : index === 1 ? 'center' : 'right'}`;
                marker.textContent = text;
                scale.appendChild(marker);
            });
*/

/*
            // Add center scale marker
            const marker = document.createElement('span');
            marker.className = "temperament-scale-marker center";
            marker.textContent = '|';
            //marker.textContent = 'Player ' + (index+1);
            scale.appendChild(marker);
*/

            //const weight = pref.weight;
            const weight = pref;

            // Add floating emoji
            const icon = document.createElement('span');
            icon.className = 'temperament-icon';
            icon.textContent = this.getEmoji(weight);
            icon.style.left = `${this.getIconPosition(weight)}%`;
            icon.style.transform = `translateX(-50%) translateY(-50%) scale(${this.getIconSize(weight)})`;
            scale.appendChild(icon);

            row.appendChild(scale);

/*                    
            // Player Number
            const name2 = document.createElement('span');
            name2.className = 'temperament-player-name';
            //name.textContent = pref.name;
            //name2.textContent = 'Player ' + (index+1);
            name2.textContent = (index+1);
            row.appendChild(name2);
*/                    

/*
            // Preference value
            const value = document.createElement('span');
            value.className = 'temperament-value';
            value.textContent = (pref.weight * 100).toFixed(0) + '%';
            row.appendChild(value);
*/
            this.container.appendChild(row);
        });
    }
}