export class Observable {
    constructor() {
      this.observers = new Set();
    }
  
    subscribe(observer) {
      this.observers.add(observer);
      return () => this.observers.delete(observer); // Returns unsubscribe function
    }
  
    notify(data) {
      this.observers.forEach(observer => observer(data));
    }
  }
  
  // Example class that uses Observable
  class UserProfile extends Observable {
    constructor(name, email) {
      super();
      this._name = name;
      this._email = email;
    }
  
    // Getter and setter for name
    get name() {
      return this._name;
    }
  
    set name(value) {
      this._name = value;
      this.notify({ property: 'name', value: value });
    }
  
    // Getter and setter for email
    get email() {
      return this._email;
    }
  
    set email(value) {
      this._email = value;
      this.notify({ property: 'email', value: value });
    }
  
    // Method to update multiple properties at once
    updateProfile(updates) {
      Object.entries(updates).forEach(([key, value]) => {
        if (this.hasOwnProperty(`_${key}`)) {
          this[`_${key}`] = value;
          this.notify({ property: key, value: value });
        }
      });
    }
  }

  /*
  
  // Usage example
  const user = new UserProfile('John Doe', 'john@example.com');
  
  // Subscribe to changes
  const unsubscribe = user.subscribe(change => {
    console.log(`Property '${change.property}' changed to: ${change.value}`);
  });
  
  // Make some changes
  user.name = 'Jane Doe';  // Logs: Property 'name' changed to: Jane Doe
  user.email = 'jane@example.com';  // Logs: Property 'email' changed to: jane@example.com
  
  // Update multiple properties at once
  user.updateProfile({
    name: 'Jane Smith',
    email: 'jane.smith@example.com'
  });
  
  // Unsubscribe when no longer needed
  unsubscribe();
  
  // This change won't trigger any notifications since we unsubscribed
  user.name = 'John Smith';

  */