import { TreatsTable } from './TreatsTable.js';
import { 
	DEBUG_DISPLAY_TREAT_TABLE
} from './GameConstants.js';

export class Treats {

    debug = true;

    treats = [];

    // instantiated values
    totalPlayers = null;
   
    // global config values
    treatsPerPlayer = null;
    treatsTotalValuePerPlayer = null;

    treatDisplayDOM = null;

    // DEPRECIATE
    currentPlayerIndex = null;

    // do we still need this?
    selectedTreat = null;

    constructor(parms) {

        this.totalPlayers = parms.totalPlayers;

        this.treatDisplayDOM = parms.treatDisplayDOM;

        // get values from the global config
        this.treatsPerPlayer = window.config.treatsPerPlayer;
        this.treatsTotalValuePerPlayer = window.config.treatsTotalValuePerPlayer;
        
        // TODO: here we've hardcoded that player 1 should always begin
        //this.currentPlayerIndex = 1; 

        this.treats = this.selectTreats(
            this.totalPlayers,
            this.treatsPerPlayer,
            this.treatsTotalValuePerPlayer
        );

        console.log(this.treats);

    }

    // an array of all the treats we'll use in this game
    treatCollection = [
        "Apple", "Avocado", "Ball of Chocolate", "Beets", "Blueberry Tea", "Burger", "Candy Cane",
        "Cheese Quesadilla", "Cherry Pie", "Chicken Quesadilla", "Chocolate Cake Pop", "Chocolate Ice Cream",
        "Chocolate Strawberry", "Chocolate Syrup", "Cocoa Beans", "Coconut", "Coffee", "Cookie", "Cotton Candy",
        "Cup of Joe", "Cupcake", "Delightful Kabab", "Drumsticks", "Éclair", "Eggs", "Energizer Coffee",    
        "French Fries", "Fried Chicken", "Fruit Smoothie", "Fudge Slush", "Gingerbread Cookie", "Grapes",
        "Gumballs", "Herbal Tea", "Hot Fudge Mini Cakes", "Hotdog", "Jelly Beans", "Kiwi", "Lemon Lime Candy",  
        "Loaded Taco", "Lollipop", "Macaroons", "Milkshake", "Muffin", "Nachos", "Noodles", "One Layer Cherry Cake",
        "Onion Rings", "Passion Fruit", "Pie", "Piña Colada", "Pineapple", "Pizza", "Plum", "Pot Stickers",
        "Rainbow Sundae", "Red Cherries", "Rice Cakes", "Strawberry Donut", "Strawberry Jam", "Strawberry",
        "Sun Soda", "Sunset Pancake", "Sweet Honey", "Toaster Pastry", "Turkey Leg", "Twisty Candy",
        "Vanilla Cake", "Watermelon", "Watermelon Iced Tea", "Green Tea", "Bananas", "Bread", "Candy Apple",
        "Cheese", "Chocolate Bar", "Chocolate Waffle", "Corn" ,"Crazy Candy Twirl", "Doughnut Cake",
        "Fish Cracker", "Fruit Juice", "Green Jelly", "Gummy Bear", "Herbal Tea", "Honey", "Hot Tea",
        "Lemon", "Marshmallow", "Nice Rice", "Orange Juice", "Popsicle", "Sour Candy", "Sugar Coffee",
        "Sweet Pancakes", "Tater Tots", "Two Layer Fudge Cake", "Veggie Sandwich"
    ];


    getCurrentTreatObj() {

        return this.selectedTreat;

    }

    // when treats is provided, use that data,
    // otherwise, use the version in our collection
    updateTreatsTableForDebug(treats = null) {

        if(DEBUG_DISPLAY_TREAT_TABLE === true) {

            console.log('updated treat table');

            const treatsTable = new TreatsTable()
                .setContainer('extended-info')
                .setTreats(treats || this.treats)
                .render();

        }
        
    }

    /*******************************************************************************
     * @function playersWithTreatsRemaining
     *
     * @description Gathers a list of players who have still have treats remaining,
     *   optionally, excepting a player who will be losing their treats.
     * 
     * @details
     * - A player loses their treats when they receive maxBites,
     *   they are then essentially out of the game.
     * - The lost treats will be randomly given to the remaining players.
     * 
     * @param {integer} optional playerIndexLosingTreats - an index of the player who
     *   will be losing their treats
     * @returns {array} an array of players who will gain the treats
     * 
     * @example
     * playersWithTreatsRemaining(playerIndexLosingTreats);
	 * 
	 * @see 
     * 
     * @author Kristi
     * @since 2025-01-06
     * @lastModified 2025-01-06
     ******************************************************************************/

    playersWithTreatsRemaining(playerIndexLosingTreats = null) {
        return this.treats.reduce((acc, playerTreats, index) => {
            if (playerTreats.length > 0 && (playerIndexLosingTreats === null || index !== playerIndexLosingTreats)) {
                acc.push(index);
            }
            return acc;
        }, []);
    }

    // https://claude.ai/chat/a9f39ac9-bdd7-4423-9bab-13b14d25f26f
    redistributePlayerTreats(playerIndexLosingTreats, playersGainingTreatsArray) {
        // Validation
        if (playerIndexLosingTreats < 0 || playerIndexLosingTreats >= this.totalPlayers) {
            throw new Error(`Invalid playerIndexLosingTreats: ${playerIndexLosingTreats}. Must be between 0 and ${this.totalPlayers - 1}`);
        }
    
        if (!Array.isArray(playersGainingTreatsArray) || playersGainingTreatsArray.length === 0) {
            throw new Error('playersGainingTreatsArray must be a non-empty array');
        }
    
        for (const playerIndex of playersGainingTreatsArray) {
            if (playerIndex < 0 || playerIndex >= this.totalPlayers) {
                throw new Error(`Invalid player index in playersGainingTreatsArray: ${playerIndex}. Must be between 0 and ${this.totalPlayers - 1}`);
            }
            if (playerIndex === playerIndexLosingTreats) {
                throw new Error(`Player losing treats (${playerIndexLosingTreats}) cannot also be in playersGainingTreatsArray`);
            }
        }
    
        if (this.treats[playerIndexLosingTreats].length === 0) {
            throw new Error(`Player ${playerIndexLosingTreats} has no treats to redistribute`);
        }
    
        const treatsToRedistribute = [...this.treats[playerIndexLosingTreats]];
        this.treats[playerIndexLosingTreats] = [];
        
        while (treatsToRedistribute.length > 0) {
            const randomTreatIndex = Math.floor(Math.random() * treatsToRedistribute.length);
            const treatToGive = treatsToRedistribute.splice(randomTreatIndex, 1)[0];
            
            const randomPlayerIndex = Math.floor(Math.random() * playersGainingTreatsArray.length);
            const playerGainingTreat = playersGainingTreatsArray[randomPlayerIndex];
            
            this.treats[playerGainingTreat].push(treatToGive);
        }
    
        // Sort treats for each player by value (greatest to smallest)
        for (let i = 0; i < this.totalPlayers; i++) {
            if (this.treats[i].length > 0) {
                this.treats[i].sort((a, b) => b.value - a.value);
            }
        }

        // if treat display debug set in GameConstants, update the DOM
        this.updateTreatsTableForDebug();            

    }

    /*******************************************************************************
	function generateTreatValues()
	
	Generates an array of random numbers divisible by 5.
		
	@param  numValues   the number of values to generate
    @param  totalValue  the total value of all the numbers generated
	                  
	Returns: an array of numbers
		 
	******************************************************************************/	
    generateTreatValues(numValues, totalValue) {

        let initialValues = [];

        // Note: If these values are altered, use the function validateGenerateTreatValues()
        // to check that a zero value isn't being returned
        const min = 15;
        const max = 100;

        // generate initial random values
        for (let i = 0; i < numValues; i++) {

            //let value = Math.random() * (max/5 - min/5) + min/5;
            let value = Math.floor(Math.random() * (max/5 - min/5) + min/5) * 5;

            initialValues.push(value);

        }
    
        //console.log(initialValues);

        // Calculate the current total
        const currentTotal = initialValues.reduce((sum, value) => sum + value, 0);
        //console.log('currentTotal: ' + currentTotal);

        // scale the values to ensure they total exactly totalValue
        const scaleFactor = totalValue / currentTotal;
        //console.log('scaleFactor: ' + scaleFactor);

        // create a new array with the scaled values
        let scaledValues = initialValues.map(weight => {
            const scaledWeight = weight * scaleFactor;
            return Math.round(scaledWeight / 5) * 5;
        });

        // sort the scaled values
        scaledValues.sort((a, b) => b - a);

        // calculate the total of all values
        const totalValueCalculated = scaledValues.reduce((sum, weight) => sum + weight, 0);
        //console.log('totalValueCalculated: ' + totalValueCalculated);

        // calculate the adjustment needed due to rounding
        let adjustment = totalValue - totalValueCalculated;
        //console.log('adjustment: ' + adjustment);

        // make the adjustment
        scaledValues[0] = scaledValues[0] + adjustment;

        return scaledValues;

    } // end generateTreatValues()

    /*******************************************************************************
	function validateGenerateTreatValues()
	
	Validates the generateTreatValues() function above.

    Logs to the console each set and detailed errors if conditions aren't met.
		
	@param  numValues   the number of values to generate
    @param  totalValue  the total value of all the numbers generated
	@param  iterations  the number of iterations, default = 100
    
	Returns: true if successful, false if unsuccessful
		 
	******************************************************************************/	    
    validateGenerateTreatValues(numValues, totalValue, iterations = 100) {

        let errorString = null;

        for (let i = 0; i < iterations; i++) {

            const treatValuesTemp = this.generateTreatValues(numValues, totalValue);
            console.log('treatValuesTemp: ' + treatValuesTemp);

            // validate the only numValues were returned
            if (treatValuesTemp.length != numValues) {

                let errorStringTemp = "numValues = " + treatValuesTemp.length;
                errorString += errorStringTemp + " | "
                console.log('ERROR: ' + errorStringTemp);

            }

            // validate that totalValue is correct
            const totalValueCalculated = treatValuesTemp.reduce((sum, weight) => sum + weight, 0);
            console.log('totalValueCalculated: ' + totalValueCalculated);
            if (totalValueCalculated != totalValue) {

                let errorStringTemp = "totalValue = " + totalValue;
                errorString += errorStringTemp + " | "
                console.log('ERROR: ' + errorStringTemp);

            }

            // validate that none of the values are 0
            const noZeros = treatValuesTemp.every(num => num !== 0);
            if (noZeros != true) {

                let errorStringTemp = "one or more zeros";
                errorString += errorStringTemp + " | "
                console.log('ERROR: ' + errorStringTemp);

            }

        }

        // display success or failure
        if(errorString === null) {

            console.log('Success!');
            return(true);
             
        } else {

            console.log('Failed! ' + errorString);
            return(false);
        }

    } // end validateGenerateTreatValues()

    /*******************************************************************************
	function shuffleTreats()
	
	Shuffle the treats array using the Fisher-Yates shuffle algorithm.

    @param  none
    
	Returns: nothing
		 
	******************************************************************************/	 
    shuffleTreats() {

        const shuffled = [...this.treatCollection];
        
        for (let i = shuffled.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [shuffled[i], shuffled[j]] = [shuffled[j], shuffled[i]];
        }
        
        this.treatCollection = shuffled;

        //console.log('shuffled: ' + this.treatCollection);

    } // end shuffleTreats()

    /*******************************************************************************
	function selectTreats()
	
	Generates an array of random numbers divisible by 5.
		
	@param  numPlayers          the number of players
    @param  numTreatsPerPlayer  the number of treats per player
	@param  totalValue          the total numeric value of all treats
    
	Returns: an array of with an object for each player

    Code Sample:
    
        let treats = new Treats();
        let myTreats = treats.selectTreats(3,5,250);

        // display Player 1's treats
        console.log(myTreats[0]);
		 
	******************************************************************************/	
    selectTreats(numPlayers, numTreatsPerPlayer, totalValue) {

        // shuffle the treats before we begin
        this.shuffleTreats();

        let treatValues = [];

        // start with the first treat in the shuffled array
        let treatIndex = 0;
        const totalTreats = this.treatCollection.length;

        let allSelectedTreats = [];

        // generate an array of treat values for each player
        for (let i = 0; i < numPlayers; i++) {
            let treatValuesSet = this.generateTreatValues(numTreatsPerPlayer,totalValue);

            //console.log('player: ' + i);
            //console.log('treatValuesSet returned: ' + treatValuesSet);
            //console.log('Total Value:', treatValuesSet.reduce((sum, weight) => sum + weight, 0));

            let playerSelectedTreats = [];

            // sequentially assign a different treat to each value
            for (let x = 0; x < numTreatsPerPlayer; x++) {

                // assign a treat to this value
                let treatName = this.treatCollection[treatIndex];
                treatIndex++;

                // if we've reached the last treat, reset the index
                if(treatIndex >= totalTreats) {

                    treatIndex = 0;

                }
                
                // prepare an object
                const myTreatObject = 
                {
                    value: treatValuesSet[x], // treatValues.index(x),
                    name: treatName
                };

                // collect the treats into our array
                playerSelectedTreats.push(myTreatObject);

            }

            allSelectedTreats.push(playerSelectedTreats);
        }

        //console.log('allSelectedTreats: ' + JSON.stringify(allSelectedTreats));

        // if treat display debug set in GameConstants, update the DOM
        this.updateTreatsTableForDebug(allSelectedTreats);

        return allSelectedTreats;

    } // end selectTreats()


    /*******************************************************************************
	function getRandomTreat()
	
	Randomly selects a treat from the selected player's collection.
		
	@param  playerIndex          the index of the desired player
    
	Returns: an array with a single treat or null if no further treats remain

    ******************************************************************************/	
    getRandomTreat(playerIndex) {

        // validate the player index
        if(playerIndex <= 0 || playerIndex > this.totalPlayers) {

            console.log('getTreat() called with an invalid playerIndex = ' + playerIndex);
            return(null);

        }

        // randomly select a treat from this player's collection
        let treatsRemaining = this.treats[playerIndex-1].length;

        if(treatsRemaining == 0) {

            return(null);
        }

        let randomTreatIndex = Math.floor(Math.random() * treatsRemaining);

        console.log('randomTreatIndex = ' + randomTreatIndex);

        let playerTreats = this.treats[playerIndex-1];
        console.table(playerTreats);

        // retrieve and remove the treat from the player's collection
        let treat = playerTreats.splice(randomTreatIndex,1);
        console.table(treat);

        console.table(playerTreats);

        // put the rest of the treats back into the collection
        this.treats[playerIndex-1] = playerTreats;
        console.log(JSON.stringify(this.treats));

        // if treat display debug set in GameConstants, update the DOM
        this.updateTreatsTableForDebug();        

        return(treat);

    } // end getRandomTreat()

    /*******************************************************************************
	function getTreatByIndex()
	
	Returns the treat matching the treatIndex from the current player's collection.
		
	@param  treatIndex          the index of the desired treat
    
	Returns: an array with a single treat or null if the treat wasn't found

    ******************************************************************************/	
    getTreatByIndex(treatIndex, playerIndex) {

        // cast as an integer
        treatIndex = parseInt(treatIndex); 

        // validate playerIndex
        if(this.validatePlayerIndex(playerIndex) === null) {

            console.log('getTreatByIndex() tried to use an invalid playerIndex = ' + playerIndex);
            return(null);

        } else {

            console.log('getTreatByIndex() called with treatIndex = ' + treatIndex + ', while currentPlayerIndex = ' + playerIndex);

            // how many treats are remaining for this player
            let treatsRemaining = this.treats[playerIndex].length;

            // handle when there are no remaining treats for this player
            if(treatsRemaining == 0) {

                console.log('getTreatByIndex() called but no treats remain for this player');
                return(null);
            }

            // validate the treatIndex
            if(treatIndex < 0 || treatIndex > treatsRemaining) {

                console.log('getTreatByIndex() tried to use an invalid treatIndex = ' + treatIndex);
                return(null);
            }         

            console.log('treatIndex = ' + treatIndex);

            // get all the treats for this player
            let playerTreats = this.treats[playerIndex];
            console.table(playerTreats);

            // retrieve and remove the treat from the player's collection
            let treat = playerTreats.splice(treatIndex,1);
            console.table(treat);

            console.table(playerTreats);

            // put the rest of the treats back into the collection
            this.treats[playerIndex] = playerTreats;
            console.log(JSON.stringify(this.treats));

            // if treat display debug set in GameConstants, update the DOM
            this.updateTreatsTableForDebug();            

            return(treat);            

        }      

    } // end getRandomTreat()

    treatsRemaining() {

        return this.treats.some(subarray => subarray.length > 0);

    }

    playerHasTreats(playerIndex) {

        // validate the playerIndex
        if((playerIndex = this.validatePlayerIndex(playerIndex)) === null) {

            console.trace(`ERROR: playerHasTreats() was passed an invalid playerIndex`);
            return false;

        } else {

            // check if this play has any treats remaining
            let treatsRemaining = this.treats[playerIndex].length;
            
            if(treatsRemaining == 0) {

                return false;

            } else {

                return true;

            }

        }

    }

    /*******************************************************************************
	function getTreatFilename()
	
	Returns the local filename of the treat when provided a display name.

    Conversions:

    * Spaces are converted to dashes
    * Lower cased
    * Foreign characters are converted and/or removed
		
	@param  treatName          the display name of the treat
    
	Returns: a local filename

    ******************************************************************************/	
    getTreatFilename(treatName) {

        //console.log("getTreatFilename() called with: " +treatName);

        // convert spaces to dashes and to lowercase
        let preparedTreatName = treatName.replace(/\s+/g, '-').toLowerCase();

        // return null if empty string
        if(preparedTreatName.length === 0) {

            console.log('getTreatFilename() has an empty string after conversion, provided treatName = ' + treatName);
            return(null);

        }

        // convert foreign characters
        // decomposes the accented characters into their base character and separates diacritical marks (accents)
        // then matches all combining diacritical marks in the Unicode range and replaces them with empty strings
        preparedTreatName = preparedTreatName.normalize('NFD').replace(/[\u0300-\u036f]/g, '');

        // return null if empty string
        if(preparedTreatName.length === 0) {

            console.log('getTreatFilename() has an empty string after conversion, provided treatName = ' + treatName);
            return(null);

        }

        // add the path and extension
        preparedTreatName = "assets/treats/" + preparedTreatName + ".png";    

        return(preparedTreatName);

    }


    /*******************************************************************************
     * @function validatePlayerIndex
     *
     * @description Validates that the provided index is within bounds
     * 
     * @details
     * - Updates the treat count in <div id="playerTreats">
     * - Refreshes any treat-related status effects
     * - Triggers animations for treat changes
     * - Updates related UI elements (e.g., shop availability)
     * 
     * @param {number} playerIndex - an index of one of the players
     * @returns {boolean} True if playerIndex is valid (1-4), false otherwise 
     * 
     * @example
     * validatePlayerIndex(playerIndex);  // call to validate the index 
     * 
     * @author Kristi
     * @since 2025-01-05
     * @lastModified 2025-01-05
     ******************************************************************************/

    validatePlayerIndex(playerIndex) {

        // convert to number if it's a string
        const num = Number(playerIndex);
        
        if (!isNaN(num) && num >= 1 && num <= 4) {
            return num;
        }
        
        console.log(`ERROR: an invalid playerIndex was passed = `, playerIndex);
        return null;

    }


    /*******************************************************************************
	function preparePlayerTreats()
	
	Updates the <div id="playerTreats"> with treats and info.
		
	@param  none          
    
	Returns: nothing

    ******************************************************************************/	

    preparePlayerTreats(playerIndex) {

        // validate the playerIndex
        if((playerIndex = this.validatePlayerIndex(playerIndex)) === null) {

            console.trace(`ERROR: displayPlayerTreats() was passed an invalid playerIndex`);
            return null;

        };

        const playerTreats = this.treats[playerIndex];
        console.log(`preparePlayerTreats() working with playerIndex = ${playerIndex}, and treats = `,playerTreats);

        const treatWrapper = document.createElement('div');
        treatWrapper.className = "treatWrapper";

        // prepare the header for the treat selection box
        const treatHeader = document.createElement('h1');
        treatHeader.className = "treatHeader";
        treatHeader.textContent = "Select a treat to feed:";
        treatWrapper.appendChild(treatHeader);

        let treatIndex = 0;

        // dynamically create links
        playerTreats.forEach(item => {

            // Example:
            // item = {value: 55, name: "Lollipop"}

            const treatName = item.name;
            const treatValue = item.value;

            // append the treatIndex
            item.id = treatIndex;

            // create the <div> for the row
            const treatRow = document.createElement('div');
            treatRow.className = "treatRow";
/*
            // create a link element
            const link = document.createElement("a");
            link.href = "#";
            //link.textContent = treatName;
            link.style.textDecoration = 'none';
            link.style.marginRight = "10px";
*/

            // add click event listener
            treatRow.addEventListener("click", (event) => {
                //event.preventDefault(); // prevent the default link behavior
                //this.handleTreatTap(item);
                treatTapped(item);
            });

            // create the treat value node
            const treatValueNode = document.createElement('span');
            treatValueNode.className = "treatValue";
            treatValueNode.textContent = treatValue;

            // create the image element
            const img = document.createElement('img');
            img.src = this.getTreatFilename(treatName);
            img.className = "treatIcon";
            img.alt = treatName;

            // create the treat name node
            const treatNameNode = document.createElement('span');
            treatNameNode.className = "treatName";
            treatNameNode.textContent = treatName;

            // append image and text to the link
            treatRow.appendChild(treatValueNode);
            treatRow.appendChild(img);
            treatRow.appendChild(treatNameNode);

            //link.appendChild(treatRow);            

            treatWrapper.appendChild(treatRow);

            // increment the treat index
            treatIndex++;
            
        });

        // return what we've prepared here 
        return treatWrapper;

/*
        let html = '';
        const width = null;
        let widthAttribute = '';
        let heightAttribute = '';
        
        for (const treatName of this.treatCollection) {
            
            const imgTag = this.prepareTreatImgTagHTML(treatName, width, height);
                        
            html = html + `
                <div style="padding-top:10px;">
                    ${imgTag} <span style="padding-left:5px;">${treatName}</span>
                </div>
            `;
            
        }

        const messageArea = document.getElementById("extended-info");
        messageArea.innerHTML = html;
*/        

    } /* end displayPlayerTreats() */

    /*******************************************************************************
	function displayOneTreat()
	
	Updates the <div id="playerTreats"> with a single treat
    and prompts the player to scan an animal NFC tag
		
	@param  treat array = {value: 60, name: "Nachos"}
    
	Returns: nothing

    ******************************************************************************/	

    displayOneTreat(treat, currentPlayerIndex) {

        //console.log('displayOneTreat() called with ' + JSON.stringify(treat));
        console.log(`displayOneTreat() called with `,treat);

        if(!this.treatDisplayDOM) {

            console.log(`[ERROR] treats.displayOneTreat() doesn't know where in the DOM to display the treat `);
            return false;

        } else {

            
            const messageArea = document.getElementById(this.treatDisplayDOM);

            if(!messageArea) {

                console.log(`[ERROR] treats.displayOneTreat() couldn't locate a DOM element with id = ${this.treatDisplayDOM}`);
                return false;

            } else {

                //const playerTreats = this.treats[playerIndex-1];

                // jump to the top of the screen
                window.scrollTo({ top: 0, behavior: 'auto' });

                const treatWrapper = document.createElement('div');
                treatWrapper.className = "treatWrapper";

                const treatName = treat[0].name;
                const treatFilename = this.getTreatFilename(treatName);

/*              PREVIOUS VERSION  
                // prepare the header for the treat selection box
                const treatHeader = document.createElement('h1');
                treatHeader.className = "treatHeader";
                treatHeader.textContent = "Scan an animal to feed:";
                treatWrapper.appendChild(treatHeader);

                // treat = {value: 55, name: "Lollipop"}

                //console.log('displayOneTreat() treat is currently = ' + JSON.stringify(treat));
                //console.log("treatName = " + treatName);
                const treatValue = treat[0].value;

                // create the <div> for the treat box
                const treatBox = document.createElement('div');
                treatBox.className = "treatBox";

                // place a large image of the treat in the box
                const img = document.createElement('img');
                img.src = this.getTreatFilename(treatName);
                img.className = "treatImage";
                img.alt = treatName;
                treatWrapper.appendChild(img);

                // place the description of the treat below it
                const treatNameNode = document.createElement('h2');
                treatNameNode.className = "treatName";
                treatNameNode.textContent = treatName;
                treatWrapper.appendChild(treatNameNode);
*/
                const markup = `
                    <div class="treatCardLargeWrapper">
                        <div class="treatCardLargeBox">
                            <h3 class="treatCardLargeHeader">Who Will You Feed?</h3>
                            <div class="treatCardLargeImageWrapper">
                                <img src="${treatFilename}" 
                                    alt="${treatName}"
                                    class="treatCardLargeImage">
                            </div>
                            <p class="treatCardLargeName">${treatName}</p>
                        </div>
                    </div>
                `;

                // update the DOM
                const treatsContainer = document.getElementById("playerTreats");
                treatsContainer.innerHTML = markup;
                //treatsContainer.appendChild(treatWrapper);

                return true;

            }

        }
        
    } /* end displayOneTreat() */    

    /*******************************************************************************
	function handleTreatTap()
	
	Is called when one of the player's treats is tapped or clicked.

    We'll replace the playerTreats <div> with the selected treat and
    then we'll wait for an animal NFC to be scanned
    or one to be input manually from the form
		
	@param  treatIDName   the ID and name of the treat that was clicked on
    
	Returns: nothing

    ******************************************************************************/	
    handleTreatTap(treatObj, currentPlayerObj) {

        // Example: {value: 60, name: "Nachos", id: "3"}

        //alert("Treat Clicked: " + JSON.stringify(treatObject));

        // get the treat from the player's collection

        console.log(`treats.handleTreatTap() handling a treat tap `,treatObj,currentPlayerObj);

        let treat = this.getTreatByIndex(treatObj.id, currentPlayerObj.number-1);
        console.log(`treats.handleTreatTap() treat = `,treat);

        // save a copy of selectedTreat
        this.selectedTreat = treat;

        // Example: treat = {value: 60, name: "Nachos"}
        const success = this.displayOneTreat(treat, currentPlayerObj.number-1);

        return success;

    } /* end handleTreatTap() */

    /*******************************************************************************
	function displayAllTreats()
	
	Prepares the HTML and displays all the treats with name and images
		
	@param  none          
    
	Returns: nothing

    ******************************************************************************/	
    displayAllTreats(height = null) {

        let html = '';
        const width = null;
        let widthAttribute = '';
        let heightAttribute = '';
        
        for (const treatName of this.treatCollection) {
            
            const imgTag = this.prepareTreatImgTagHTML(treatName, width, height);
                        
            html = html + `
                <div class="treatCollection" style="padding-top:10px;">
                    ${imgTag} <span style="padding-left:5px;">${treatName}</span>
                </div>
            `;
            
        }

        const messageArea = document.getElementById("extended-info");
        messageArea.innerHTML = html;

    } /* end displayAllTreats() */

    /*******************************************************************************
	function prepareTreatImgTag()

    DEPRECIATED
	
	Prepares the HTML image tag for the specified treat name
		
	@param  treatName   the display name of the treat          
    
	Returns: prepared HTML code for the img tag

    ******************************************************************************/	
    prepareTreatImgTagHTML(treatName, width = null, height = null) {

        let html = '';
        let widthAttribute = '';
        let heightAttribute = '';
        
        // get the filename for this treat
        const filename = this.getTreatFilename(treatName);

        // prepare the width attribute
        if(width > 0) {
            widthAttribute =  `width="${width}" `;
        }

        // prepare the height attribute
        if(height > 0) {
            heightAttribute =  `height="${height}" `;
        }            

        html = `<img src="${filename}" alt="${treatName}" ${widthAttribute}${heightAttribute}/>`;
                                    
        return(html);

    } /* end prepareTreatImgTagHTML() */

    /*******************************************************************************
     * @function validatePlayerIndex
     *
     * @description Validates that the provided index is within bounds
     * 
     * @details
     * - Updates the treat count in <div id="playerTreats">
     * - Refreshes any treat-related status effects
     * - Triggers animations for treat changes
     * - Updates related UI elements (e.g., shop availability)
     * 
     * @param {number} playerIndex - an index of one of the players
     * @returns {boolean} True if playerIndex is valid (1-4), false otherwise 
     * 
     * @example
     * validatePlayerIndex(playerIndex);  // call to validate the index 
     * 
     * @author Kristi
     * @since 2025-01-05
     * @lastModified 2025-01-05
     ******************************************************************************/

    validatePlayerIndex(playerIndex) {

        // convert to number if it's a string
        const num = Number(playerIndex);
        
        if (!isNaN(num) && num >= 0 && num <= this.totalPlayers-1) {
            return num;
        }
        
        console.log(`ERROR: an invalid playerIndex was passed = `, playerIndex);
        return null;

    }


}